local mb = require("libmodbus")

--[[
local devs = {
    { adr = 1, type = "RKT-222", disconn = {2,2,2,2} },
    { adr = 2, type = "RKT-252", disconn = {1001,1001,1001,1001} },
}
]] -------------------------------------------------------------------
-- Возвращаемые значения libmodbus:
-- read_bits 0x1
-- read_input_bits 0x2
-- read_registers 0x3
-- read_input_registers 0x4
-- C++
-- должна возвращать количество прочитанных входных данных в случае успеха.
-- В противном случае он должен вернуть -1 и установить errno.
-- Lua
--  возвращает таблицу данных в случае успеха.
-- В случае ошибки nil и строку ошибки
-- write_bits 0x0F
-- write_registers 0x10
-- write_bit 0x5
-- write_register 0x6
-- C++
-- должна возвращать количество записанных данных в случае успеха.
-- В противном случае он должен вернуть -1 и установить errno.
-- Lua
--  возвращает true в случае успеха.
-- В случае ошибки nil и строку ошибки
-------------------------------------------------------------------
-- Переменная время цикла мсек
VAR("IN", "Время_цикла_опроса_мсек")

local wait = wait_from_net_and_sleep
-------------------------------------------------------------------
-- Запуск modbus
local ctx = rkt_new_rtu("internal", 9600, "N", 8, 1)

ctx:set_response_timeout(0, 150000)

local sec, mcsec = ctx:get_response_timeout()
print("timeout sec = " .. sec .. " mcsec=" .. mcsec)

-- ctx:set_debug()
ctx:set_error_recovery(mb.ERROR_RECOVERY_LINK, mb.ERROR_RECOVERY_PROTOCOL)

local ok, err = ctx:connect()
if not ok then
    sys_log_io_service("Couldn't connect: " .. err)
end

-------------------------------------------------------------------
function R(func, address, par2, dev)
    dev.req_count = dev.req_count + 1
    local res, err_msg = func(ctx, address, par2)
    wait_from_net_and_sleep(5)

    if res == nil then
        sys_log_io_service("slave=" .. dev.adr .. " ,error_handling ," .. err_msg)

        -- Процент ошибок
        dev.error_count = dev.error_count + 1
        -- set_disconnected(dev)
        print("set_disconnected", dev.type, dev.adr)
        v[dev.disconnected_name] = true
    elseif v[dev.disconnected_name] then
        -- set_connected(dev)
        print("set_connected", dev.type, dev.adr)

        -- Извещаем что модуль подключился
        v[dev.disconnected_name] = false
    end

    if (res == nil) or (dev.req_count % 1000 == 0) then
        -- Выводим  процент ошибок
        v[dev.percent_error_RS485_name] = 100 * dev.error_count / dev.req_count
    end

    return res
end

-------------------------------------------------------------------
-- Устанавливаем значения выходов при разрыве связи modbus
local function set_output_disconnect(arr, dev)
    dev.set_output_disconnect_func = function()
        -- Настройка дискретных выходов при разрыве связи modbus
        if type(dev.disconn) == "table" then
            for i = 1, 4 do
                if math.type(dev.disconn[i]) == "integer" then
                    arr[i] = dev.disconn[i]
                end
            end
        end
        ctx:set_slave(dev.adr)
        wait(30)
        local res, err_msg = ctx:write_registers(61, arr)
        if res then
            dev.set_output_disconnect_func = nil
            print(dev.type, dev.adr,
                string.format("ctx:write_registers(61, {%d,%d,%d,%d})", arr[1], arr[2], arr[3], arr[4]))
        else
            print(dev.type, dev.adr, "error set_output_disconnect_func")
        end
        wait(30)
    end
    -- dev.set_output_disconnect_func()
end

-------------------------------------------------------------------
function rkt_module(module_type)
    local o = {}
    if module_type == "RKT-222" then
        o.vars_def = function(dev)
            -- Дискретные входы
            dev.DI_var_names = {}
            for i = 1, 8 do
                dev.DI_var_names[i] = string.format("DI%d.%d.RKT-222", i, dev.adr)
                VAR("DS", dev.DI_var_names[i])
            end
            -- Дискретные выходы
            dev.DO_var_names = {}
            for i = 1, 4 do
                dev.DO_var_names[i] = string.format("DO%d.%d.RKT-222", i, dev.adr)
                VAR("DS", dev.DO_var_names[i])
            end
            -- Настройка дискретных выходов при разрыве связи modbus
            set_output_disconnect({ 2, 2, 2, 2 }, dev)
        end
    else -- RKT-232,RKT-242,RKT-252, RKT-302
        o.vars_def = function(dev)
            -- Аналоговые входы
            dev.AI_var_names = {}
            for i = 1, 8 do
                dev.AI_var_names[i] = string.format("AI%d.%d.%s", i, dev.adr, dev.type)
                VAR("AN", dev.AI_var_names[i])
            end
            -- Аналоговые выходы
            dev.AO_var_names = {}
            for i = 1, 4 do
                dev.AO_var_names[i] = string.format("AO%d.%d.%s", i, dev.adr, dev.type)
                VAR("AN", dev.AO_var_names[i])
            end
            -- Настройка аналоговых выходов при разрыве связи modbus
            set_output_disconnect({ 1001, 1001, 1001, 1001 }, dev)
        end
    end
    if module_type == "RKT-222" then
        o.cycle_read = function(dev)
            -- Чтение дискретных входов
            local arr_res = R(ctx.read_input_bits, 1, 8, dev)
            if arr_res ~= nil then
                for i = 1, 8 do
                    v[dev.DI_var_names[i]] = (arr_res[i] == 1)
                    -- print(dev.DI_var_names[i], v[dev.DI_var_names[i]])
                end
            end
        end
    elseif module_type == "RKT-252" then
        o.cycle_read = function(dev)
            -- Чтение аналоговых входов
            local arr_res = R(ctx.read_input_registers, 1, 8, dev)
            if arr_res ~= nil then
                for i = 1, 8 do
                    v[dev.AI_var_names[i]] = arr_res[i]
                    -- print(dev.AI_var_names[i], v[dev.AI_var_names[i]])
                end
            end
        end
    else -- RKT-232,RKT-242, RKT-302
        o.cycle_read = function(dev)
            -- Чтение аналоговых входов
            local arr_res = R(ctx.read_input_registers, 1, 8, dev)
            if arr_res ~= nil then
                for i = 1, 8 do
                    v[dev.AI_var_names[i]] = mb.get_s16(arr_res[i]) * 0.1
                    -- print(dev.AI_var_names[i], v[dev.AI_var_names[i]])
                end
            end
        end
    end

    if module_type == "RKT-222" then
        o.cycle_write = function(dev)
            -- Запись дискретных выходов
            local arr = {}
            for i = 1, 4 do
                arr[i] = v[dev.DO_var_names[i]]
                -- print(dev.DO_var_names[i], v[dev.DO_var_names[i]])
            end
            local res = R(ctx.write_bits, 21, arr, dev)
        end
    else -- RKT-232,RKT-242,RKT-252, RKT-302
        o.cycle_write = function(dev)
            -- Запись аналоговых выходов
            local arr = {}
            for i = 1, 4 do
                arr[i] = v[dev.AO_var_names[i]] * 10
                -- print(dev.AO_var_names[i], v[dev.AO_var_names[i]])
            end
            local res = R(ctx.write_registers, 21, arr, dev)
        end
    end
    return o
end

-------------------------------------------------------------------
local RKTs = {
    ["RKT-302"] = rkt_module("RKT-302"),
    ["RKT-222"] = rkt_module("RKT-222"),
    ["RKT-232"] = rkt_module("RKT-232"),
    ["RKT-242"] = rkt_module("RKT-242"),
    ["RKT-252"] = rkt_module("RKT-252")
}
-------------------------------------------------------------------
-- Создание служебных переменных
for i, dev in ipairs(devs) do
    if not dev.not_run then
        local postfix = "_" .. dev.adr .. "." .. dev.type
        -- Переменная обрыва модуля
        dev.disconnected_name = "Disconnect" .. postfix
        VAR("DS", dev.disconnected_name)

        dev.req_count = 0
        dev.error_count = 0

        -- Переменная количества ошибочных запрсов
        dev.percent_error_RS485_name = "Процент_ошибок_RS485" .. postfix
        VAR("AN", dev.percent_error_RS485_name)
    end
end

-- Создание переменных входов - выходов
for i, dev in ipairs(devs) do
    if not dev.not_run then
        RKTs[dev.type].vars_def(dev)
    end
end

-------------------------------------------------------------------
function run()
    local min_delay_ms = math.ceil(1000 * 3.5 * (1 + 8 + 1) / 9600)
    print("min_delay_ms:" .. min_delay_ms)

    -- Цикл делаем минимум примерно 400 мс, чтобы не слать много сообщений
    -- Цикл может быть больше, но не меньше.
    -- Для одного модуля должна быть задержка минимум 30 мс между запросами
    -- Ориентируемся по аналоговому модулю - время опроса 75+5 = 80мс
    -- delay_ms = 155, 55, 21, 5, 4, ...
    local delay_ms = math.floor((400 - #devs * 80) / (2 * #devs)) - 5
    if delay_ms < min_delay_ms then
        delay_ms = min_delay_ms
    end
    print('delay_ms=', delay_ms)
    while true do
        local start_time = now_ms()
        -- Чтение переменных
        for i, dev in ipairs(devs) do
            if not dev.not_run then
                ctx:set_slave(dev.adr)
                RKTs[dev.type].cycle_read(dev)
            end
            wait(delay_ms)
        end
        -- Запись переменных
        for i, dev in ipairs(devs) do
            if not dev.not_run then
                -- Устанавливаем, если не установлены значения выходов при разрыве связи modbus
                if dev.set_output_disconnect_func then
                    dev.set_output_disconnect_func()
                end

                ctx:set_slave(dev.adr)
                RKTs[dev.type].cycle_write(dev)
            end
            wait(delay_ms)
        end
        
        -- Проверяем наличия add_run
        if type(add_run) == "function" then
            add_run(ctx)
        end

        local cycle_time = now_ms() - start_time
        --print("Время modbus цикла  мсек :", cycle_time)
        local old_cycle = v["Время_цикла_опроса_мсек"]
        if math.abs(cycle_time - old_cycle) > 5 then
            v["Время_цикла_опроса_мсек"] = cycle_time
        end
    end
end
